% This program fills a matrix and right-hand-side vector. The matrix is
% a discretization of the Poisson equation.
%
% This program is used to fill the matrix and righthandside. This is a 
% discretization of the following equation:
%
%      -c  - c   +u1*c  + u2*c  = f  x,y in [0,1]x[0,1],
%        xx   yy      x       y
%
% with Dirichlet boundary conditions everywhere. The number of grid
% points in x-direction is n1, the number of grid points in y-direction 
% is n2. The horizontal velocity is u1 and the vertical velocity is u2.
%
% [a,f] = poisson(n1,n2,u1,u2,disc);
%
% programmer: C. Vuik, TU Delft, The Netherlands
% e-mail    : c.vuik@math.tudelft.nl
% date      : 04-06-1998
% course    : Numerieke stromingsleer II (a PhD course from the J.M
%             Burgers Centre)
%

function [a,f] = poisson(n1,n2,u1,u2,disc);
clear a f
built = 0;
m = n1*n2;
h1 = 1/(n1+1);
h2 = 1/(n2+1);
if size(double(disc)) == size(double('central'))
   if double(disc) == double('central')
      built = 1;
      for i = 1:m
         a(i,i) = 2/h1^2+2/h2^2;
      end
      for i = 1:m-n1
         a(i,i+n1) = -1/h2^2+u2/(2*h2);
         a(i+n1,i) = -1/h2^2-u2/(2*h2);
      end
      for i = 1:m-1
         a(i,i+1) = -1/h1^2+u1/(2*h1);
         a(i+1,i) = -1/h1^2-u1/(2*h1);
      end
   end
end
%
% end if loop central
%
if size(double(disc)) == size(double('upwind'))
   if double(disc) == double('upwind')
      built = 1;
      for i = 1:m
         a(i,i) = 2/h1^2+2/h2^2+abs(u1)/h1+abs(u2)/h2;
      end
      for i = 1:m-n1
         if u2 > 0
            a(i,i+n1) = -1/h2^2;
            a(i+n1,i) = -1/h2^2-u2/(h2);
         else
            a(i,i+n1) = -1/h2^2+u2/(h2);
            a(i+n1,i) = -1/h2^2;
         end
      end
      for i = 1:m-1
         if u1 > 0
            a(i+1,i) = -1/h1^2-u1/(h1);
            a(i,i+1) = -1/h1^2;
         else
            a(i,i+1) = -1/h1^2+u1/(h1);
            a(i+1,i) = -1/h1^2;
         end
      end
   end
end
%
% end if loop upwind
%

for i = 1:n2-1
   a(i*n1,i*n1+1) = 0;
   a(i*n1+1,i*n1) = 0;
end
for i = 1:m
   f(i,1) = 1;
%   f(i,1) = i*h1*h2;
end
for i = 1:n1
   for j = 1:n2
      f(i+(j-1)*n1) = 1+sin(pi*i*h1)*sin(pi*j*h2);
   end
end
%a = sparse(a);
if built == 0
   disp('Warning the matrix and right-hand-side vector has not been built')
end
